<?php
/**
 * Main plugin class
 */

if (!defined('ABSPATH')) {
    exit;
}

class AI_Hair_Simulator {
    
    private static $instance = null;
    private $admin;
    private $api;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        $this->admin = new AI_Hair_Simulator_Admin();
        $this->api = new AI_Hair_Simulator_API();
    }
    
    public function init() {
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Register shortcodes
        add_shortcode('ai_hair_simulator_button', array($this, 'render_button_shortcode'));
        
        // Initialize admin
        $this->admin->init();
        
        // Initialize API
        $this->api->init();
    }
    
    public function enqueue_scripts() {
        // Only enqueue on pages with the shortcode
        global $post;
        if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'ai_hair_simulator_button')) {
            wp_enqueue_style(
                'ai-hair-simulator-style',
                AI_HAIR_SIMULATOR_PLUGIN_URL . 'assets/css/frontend.css',
                array(),
                AI_HAIR_SIMULATOR_VERSION
            );
            
            wp_enqueue_script(
                'ai-hair-simulator-script',
                AI_HAIR_SIMULATOR_PLUGIN_URL . 'assets/js/frontend.js',
                array('jquery'),
                AI_HAIR_SIMULATOR_VERSION,
                true
            );
            
            // Localize script with settings
            wp_localize_script('ai-hair-simulator-script', 'aiHairSimulator', array(
                'apiUrl' => get_option('ai_hair_simulator_api_url', 'https://www.aihairtransplant.io'),
                'activationKey' => get_option('ai_hair_simulator_activation_key', ''),
                'website' => home_url(),
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('ai_hair_simulator_nonce'),
                'labels' => array(
                    'fullName' => __('Full Name', 'ai-hair-simulator'),
                    'email' => __('Email', 'ai-hair-simulator'),
                ),
                'i18n' => array(
                    'uploadTitle' => __('Upload Photo', 'ai-hair-simulator'),
                    'selectHairstyle' => __('Select Hairstyle (Optional)', 'ai-hair-simulator'),
                    'simulate' => __('Simulate', 'ai-hair-simulator'),
                    'tryAnother' => __('Try Another', 'ai-hair-simulator'),
                    'originalTitle' => __('Original', 'ai-hair-simulator'),
                    'newHairstyle' => __('New Hairstyle', 'ai-hair-simulator'),
                    'loading' => __('Processing...', 'ai-hair-simulator'),
                    'error' => __('An error occurred. Please try again.', 'ai-hair-simulator'),
                    'invalidKey' => __('Invalid activation key. Please contact support.', 'ai-hair-simulator'),
                )
            ));
        }
    }
    
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_ai-hair-simulator' !== $hook) {
            return;
        }
        
        // Ensure jQuery is loaded (WordPress admin usually has it, but let's be sure)
        wp_enqueue_script('jquery');
        
        wp_enqueue_style(
            'ai-hair-simulator-admin-style',
            AI_HAIR_SIMULATOR_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            AI_HAIR_SIMULATOR_VERSION
        );
    }
    
    /**
     * Render button-only shortcode that opens simulator in a modal.
     */
    public function render_button_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => __('AI Hair Simulator', 'ai-hair-simulator'),
            'label' => __('Try the AI Hair Simulator', 'ai-hair-simulator'),
        ), $atts);

        // Check activation key is set
        $activation_key = get_option('ai_hair_simulator_activation_key', '');
        if (empty($activation_key)) {
            return '<div class="ai-hair-simulator-error">' .
                __('Please configure the activation key in plugin settings.', 'ai-hair-simulator') .
                '</div>';
        }

        ob_start();
        $button_label = $atts['label'];
        include AI_HAIR_SIMULATOR_PLUGIN_DIR . 'templates/modal.php';
        return ob_get_clean();
    }
    
    public static function activate() {
        // Set default options
        if (get_option('ai_hair_simulator_api_url') === false) {
            add_option('ai_hair_simulator_api_url', 'https://www.aihairtransplant.io');
        }
        if (get_option('ai_hair_simulator_activation_key') === false) {
            add_option('ai_hair_simulator_activation_key', '');
        }

        // Create log table for simulations
        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_hair_simulator_logs';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL,
            full_name VARCHAR(255) NOT NULL,
            email VARCHAR(255) NOT NULL,
            phone VARCHAR(50) NULL,
            city VARCHAR(100) NULL,
            age INT(11) NULL,
            hairstyle VARCHAR(100) NULL,
            processed_image_url TEXT NULL,
            website VARCHAR(255) NULL,
            activation_key VARCHAR(255) NULL,
            PRIMARY KEY  (id)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }
    
    public static function deactivate() {
        // Clean up if needed
    }
}

