<?php
/**
 * API integration class
 */

if (!defined('ABSPATH')) {
    exit;
}

class AI_Hair_Simulator_API {
    
    public function init() {
        // AJAX handler to log simulations with user details
        add_action('wp_ajax_ai_hair_simulator_log', array($this, 'handle_log_simulation'));
        add_action('wp_ajax_nopriv_ai_hair_simulator_log', array($this, 'handle_log_simulation'));
    }
    
    /**
     * Validate activation key
     */
    public static function validate_key($activation_key, $website) {
        $api_url = get_option('ai_hair_simulator_api_url', '');
        
        if (empty($api_url) || empty($activation_key)) {
            return array(
                'valid' => false,
                'message' => __('API URL or activation key not configured', 'ai-hair-simulator')
            );
        }
        
        $response = wp_remote_post($api_url . '/api/activation-keys/validate', array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'activationKey' => $activation_key,
                'website' => $website
            )),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            return array(
                'valid' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['valid']) && $body['valid']) {
            return array(
                'valid' => true,
                'data' => $body
            );
        }
        
        return array(
            'valid' => false,
            'message' => isset($body['statusMessage']) ? $body['statusMessage'] : __('Invalid activation key', 'ai-hair-simulator')
        );
    }

    /**
     * Handle AJAX request to log simulation data (kept for backwards compatibility, but data is now stored in Supabase).
     * This endpoint just returns success since the data is already stored in Supabase when the simulation is created.
     */
    public function handle_log_simulation() {
        check_ajax_referer('ai_hair_simulator_nonce', 'nonce');

        $full_name = isset($_POST['fullName']) ? sanitize_text_field(wp_unslash($_POST['fullName'])) : '';
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';

        if (empty($full_name) || empty($email)) {
            wp_send_json_error(array(
                'message' => __('Full name and email are required.', 'ai-hair-simulator'),
            ));
        }

        // Data is already stored in Supabase via the /api/hair-simulations endpoint
        // This is just a compatibility endpoint that returns success
        wp_send_json_success(array(
            'message' => __('Simulation logged successfully.', 'ai-hair-simulator'),
        ));
    }
}

