<?php
/**
 * Admin settings class
 */

if (!defined('ABSPATH')) {
    exit;
}

class AI_Hair_Simulator_Admin {
    
    public function init() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    public function add_admin_menu() {
        add_options_page(
            __('AI Hair Simulator Settings', 'ai-hair-simulator'),
            __('AI Hair Simulator', 'ai-hair-simulator'),
            'manage_options',
            'ai-hair-simulator',
            array($this, 'render_settings_page')
        );
    }
    
    public function register_settings() {
        register_setting('ai_hair_simulator_settings', 'ai_hair_simulator_api_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));

        register_setting('ai_hair_simulator_settings', 'ai_hair_simulator_activation_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));

        // Optional user info fields visibility
        register_setting('ai_hair_simulator_settings', 'ai_hair_simulator_show_phone', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 1,
        ));

        register_setting('ai_hair_simulator_settings', 'ai_hair_simulator_show_city', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 1,
        ));

        register_setting('ai_hair_simulator_settings', 'ai_hair_simulator_show_age', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 1,
        ));

        add_settings_section(
            'ai_hair_simulator_main_section',
            __('Plugin Settings', 'ai-hair-simulator'),
            array($this, 'render_section_description'),
            'ai-hair-simulator'
        );
        
        add_settings_field(
            'ai_hair_simulator_api_url',
            __('API URL', 'ai-hair-simulator'),
            array($this, 'render_api_url_field'),
            'ai-hair-simulator',
            'ai_hair_simulator_main_section'
        );
        
        add_settings_field(
            'ai_hair_simulator_activation_key',
            __('Activation Key', 'ai-hair-simulator'),
            array($this, 'render_activation_key_field'),
            'ai-hair-simulator',
            'ai_hair_simulator_main_section'
        );

        add_settings_field(
            'ai_hair_simulator_optional_fields',
            __('Optional User Fields', 'ai-hair-simulator'),
            array($this, 'render_optional_fields_section'),
            'ai-hair-simulator',
            'ai_hair_simulator_main_section'
        );
    }

    /**
     * Sanitize checkbox values to 0/1.
     */
    public function sanitize_checkbox($value) {
        return $value ? 1 : 0;
    }

    public function render_section_description() {
        echo '<p>' . __('Configure your AI Hair Simulator plugin settings below.', 'ai-hair-simulator') . '</p>';
    }
    
    public function render_api_url_field() {
        $value = get_option('ai_hair_simulator_api_url', 'https://www.aihairtransplant.io');
        echo '<input type="url" name="ai_hair_simulator_api_url" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . __('Enter the API URL for the hair simulator service (e.g., https://www.aihairtransplant.io)', 'ai-hair-simulator') . '</p>';
    }
    
    public function render_activation_key_field() {
        $value = get_option('ai_hair_simulator_activation_key', '');
        $website = home_url();
        
        echo '<input type="text" name="ai_hair_simulator_activation_key" value="' . esc_attr($value) . '" class="regular-text" id="activation-key-input" />';
        echo '<button type="button" id="validate-key-btn" class="button">' . __('Validate Key', 'ai-hair-simulator') . '</button>';
        echo '<span id="key-validation-status" class="validation-status"></span>';
        echo '<p class="description">' . __('Enter your activation key. Your website URL: ', 'ai-hair-simulator') . '<strong>' . esc_html($website) . '</strong></p>';
        
        // Output script directly to ensure it runs
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            console.log('AI Hair Simulator: Validation script loaded');
            
            $('#validate-key-btn').on('click', function(e) {
                e.preventDefault();
                console.log('AI Hair Simulator: Validate button clicked');
                
                var key = $('#activation-key-input').val();
                var apiUrl = $('input[name="ai_hair_simulator_api_url"]').val();
                var website = '<?php echo esc_js($website); ?>';
                var statusEl = $('#key-validation-status');
                
                console.log('AI Hair Simulator: Key =', key ? '***' : 'empty', 'API URL =', apiUrl);
                
                if (!key || !apiUrl) {
                    statusEl.html('<span style="color: red;">⚠ <?php echo esc_js(__('Please enter both API URL and activation key', 'ai-hair-simulator')); ?></span>');
                    return;
                }
                
                // Remove trailing slash from API URL to prevent double slashes
                apiUrl = apiUrl.replace(/\/+$/, '');
                var validationUrl = apiUrl + '/api/activation-keys/validate';
                
                console.log('AI Hair Simulator: Making request to', validationUrl);
                
                statusEl.html('<span style="color: blue;">🔄 <?php echo esc_js(__('Validating...', 'ai-hair-simulator')); ?></span>');
                
                $.ajax({
                    url: validationUrl,
                    method: 'POST',
                    contentType: 'application/json',
                    crossDomain: true,
                    data: JSON.stringify({
                        activationKey: key,
                        website: website
                    }),
                    success: function(response) {
                        console.log('AI Hair Simulator: Validation success', response);
                        if (response.success && response.valid) {
                            statusEl.html('<span style="color: green;">✅ <?php echo esc_js(__('Key is valid', 'ai-hair-simulator')); ?></span>');
                        } else {
                            statusEl.html('<span style="color: red;">❌ <?php echo esc_js(__('Invalid key', 'ai-hair-simulator')); ?></span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AI Hair Simulator: Validation error', xhr, status, error);
                        var message = '❌ <?php echo esc_js(__('Validation failed', 'ai-hair-simulator')); ?>';
                        if (xhr.responseJSON && xhr.responseJSON.statusMessage) {
                            message += ': ' + xhr.responseJSON.statusMessage;
                        } else if (xhr.status === 0) {
                            message += ': CORS error or network issue. Check browser console.';
                        } else if (xhr.statusText) {
                            message += ' (' + xhr.status + ': ' + xhr.statusText + ')';
                        } else if (error) {
                            message += ' (' + error + ')';
                        }
                        statusEl.html('<span style="color: red;">' + message + '</span>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /**
     * Render checkboxes for optional user fields.
     */
    public function render_optional_fields_section() {
        $show_phone  = get_option('ai_hair_simulator_show_phone', 1);
        $show_city   = get_option('ai_hair_simulator_show_city', 1);
        $show_age    = get_option('ai_hair_simulator_show_age', 1);
        ?>
        <fieldset>
            <legend class="screen-reader-text">
                <span><?php esc_html_e('Optional User Fields', 'ai-hair-simulator'); ?></span>
            </legend>
            <label>
                <input type="checkbox" name="ai_hair_simulator_show_phone" value="1" <?php checked(1, $show_phone); ?> />
                <?php esc_html_e('Phone number', 'ai-hair-simulator'); ?>
            </label>
            <br/>
            <label>
                <input type="checkbox" name="ai_hair_simulator_show_city" value="1" <?php checked(1, $show_city); ?> />
                <?php esc_html_e('City', 'ai-hair-simulator'); ?>
            </label>
            <br/>
            <label>
                <input type="checkbox" name="ai_hair_simulator_show_age" value="1" <?php checked(1, $show_age); ?> />
                <?php esc_html_e('Age', 'ai-hair-simulator'); ?>
            </label>
            <p class="description">
                <?php esc_html_e('Full name and email are always required. These fields control which additional visitor details are shown (but remain optional for the visitor).', 'ai-hair-simulator'); ?>
            </p>
        </fieldset>
        <?php
    }
    
    /**
     * Fetch simulations from Supabase API
     */
    private function fetch_simulations_from_api() {
        $activation_key = get_option('ai_hair_simulator_activation_key', '');
        $api_url = get_option('ai_hair_simulator_api_url', '');
        $website = home_url();
        
        if (empty($activation_key) || empty($api_url)) {
            return array('success' => false, 'data' => array());
        }
        
        // Remove trailing slash from API URL
        $api_url = rtrim($api_url, '/');
        
        $response = wp_remote_get($api_url . '/api/clinic-simulations?activationKey=' . urlencode($activation_key) . '&website=' . urlencode($website), array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'data' => array(), 'error' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200 && isset($body['data'])) {
            return array('success' => true, 'data' => $body['data']);
        }
        
        return array('success' => false, 'data' => array(), 'error' => isset($body['statusMessage']) ? $body['statusMessage'] : 'Unknown error');
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        if (isset($_GET['settings-updated'])) {
            add_settings_error('ai_hair_simulator_messages', 'ai_hair_simulator_message', __('Settings Saved', 'ai-hair-simulator'), 'updated');
        }
        
        settings_errors('ai_hair_simulator_messages');
        
        $activation_key = get_option('ai_hair_simulator_activation_key', '');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php if (empty($activation_key)) : ?>
            <div class="notice notice-warning" style="padding: 15px; margin: 20px 0; border-left: 4px solid #f59e0b;">
                <h2 style="margin-top: 0;"><?php _e('⚠️ Activation Key Required', 'ai-hair-simulator'); ?></h2>
                <p style="font-size: 15px;">
                    <?php _e('To use the AI Hair Transplant Simulator plugin, you need an active license.', 'ai-hair-simulator'); ?>
                </p>
                <p style="font-size: 14px;">
                    <strong><?php _e('Get your license at:', 'ai-hair-simulator'); ?></strong><br>
                    <a href="https://www.aihairtransplant.io/clinic" target="_blank" rel="noopener noreferrer" style="font-size: 16px; color: #3b82f6; text-decoration: none; font-weight: 600;">
                        https://www.aihairtransplant.io/clinic
                    </a>
                </p>
                <p style="font-size: 14px; color: #6b7280;">
                    <?php _e('Contact us to get your activation key and start capturing leads today.', 'ai-hair-simulator'); ?>
                </p>
            </div>
            <?php endif; ?>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('ai_hair_simulator_settings');
                do_settings_sections('ai-hair-simulator');
                submit_button(__('Save Settings', 'ai-hair-simulator'));
                ?>
            </form>
            
            <div class="ai-hair-simulator-info">
                <h2><?php _e('Usage', 'ai-hair-simulator'); ?></h2>
                <p><?php _e('To display the simulator, use the button shortcode which opens it in a modal:', 'ai-hair-simulator'); ?></p>
                <code>[ai_hair_simulator_button label="Try the AI Hair Simulator" title="AI Hair Simulator"]</code>
            </div>

            <?php
            // Fetch simulations from Supabase API
            if (!empty($activation_key)) {
                $result = $this->fetch_simulations_from_api();
                $simulations = $result['success'] ? $result['data'] : array();
            } else {
                $simulations = array();
            }
            ?>
            <div class="ai-hair-simulator-logs">
                <h2><?php _e('Recent Simulations', 'ai-hair-simulator'); ?></h2>
                <?php if (!empty($activation_key)) : ?>
                    <?php if (!empty($simulations)) : ?>
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Date', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Full Name', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Email', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Phone', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('City', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Age', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Hairstyle', 'ai-hair-simulator'); ?></th>
                                    <th><?php _e('Images', 'ai-hair-simulator'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($simulations as $simulation) : ?>
                                <tr>
                                    <td><?php echo esc_html(date('Y-m-d H:i', strtotime($simulation['created_at']))); ?></td>
                                    <td><?php echo esc_html($simulation['full_name'] ?? '—'); ?></td>
                                    <td>
                                        <?php if (!empty($simulation['email'])) : ?>
                                            <a href="mailto:<?php echo esc_attr($simulation['email']); ?>"><?php echo esc_html($simulation['email']); ?></a>
                                        <?php else : ?>
                                            —
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo esc_html($simulation['phone'] ?? '—'); ?></td>
                                    <td><?php echo esc_html($simulation['city'] ?? '—'); ?></td>
                                    <td><?php echo esc_html($simulation['age'] ?? '—'); ?></td>
                                    <td><?php echo esc_html($simulation['hairstyle_selected'] ?? '—'); ?></td>
                                    <td>
                                        <?php if (!empty($simulation['original_image_url'])) : ?>
                                            <a href="<?php echo esc_url($simulation['original_image_url']); ?>" target="_blank" rel="noopener noreferrer" download style="margin-right: 8px;">
                                                <?php _e('📥 Before', 'ai-hair-simulator'); ?>
                                            </a>
                                        <?php endif; ?>
                                        <?php if (!empty($simulation['processed_image_url'])) : ?>
                                            <a href="<?php echo esc_url($simulation['processed_image_url']); ?>" target="_blank" rel="noopener noreferrer" download>
                                                <?php _e('📥 After', 'ai-hair-simulator'); ?>
                                            </a>
                                        <?php endif; ?>
                                        <?php if (empty($simulation['original_image_url']) && empty($simulation['processed_image_url'])) : ?>
                                            —
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                        <p class="description">
                            <?php _e('Showing the most recent simulations from your clinic. Data is fetched from the cloud database.', 'ai-hair-simulator'); ?>
                        </p>
                    <?php else : ?>
                        <?php if ($result['success'] === false && isset($result['error'])) : ?>
                            <div class="notice notice-error inline">
                                <p><?php echo esc_html(sprintf(__('Error fetching simulations: %s', 'ai-hair-simulator'), $result['error'])); ?></p>
                            </div>
                        <?php else : ?>
                            <p><?php _e('No simulations logged yet. When customers use the simulator, they will appear here.', 'ai-hair-simulator'); ?></p>
                        <?php endif; ?>
                    <?php endif; ?>
                <?php else : ?>
                    <p><?php _e('Please enter and save your activation key above to view simulations.', 'ai-hair-simulator'); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}

