(function($) {
    'use strict';

    const HairSimulator = {
        apiUrl: '',
        activationKey: '',
        website: '',
        selectedImage: null,
        previewImageUrl: null,
        selectedHairstyle: 'autodetect',
        processedImageUrl: null,
        activeSimulationId: null,
        fullName: '',
        email: '',
        phone: '',
        city: '',
        age: '',
        consentMarketing: false,

        init: function() {
            if (typeof aiHairSimulator === 'undefined') {
                console.error('AI Hair Simulator: Configuration not found');
                return;
            }

            // Remove trailing slash from API URL to prevent double slashes
            this.apiUrl = (aiHairSimulator.apiUrl || '').replace(/\/+$/, '');
            this.activationKey = aiHairSimulator.activationKey;
            this.website = aiHairSimulator.website;

            if (!this.apiUrl || !this.activationKey) {
                this.showError('Please configure the plugin settings in WordPress admin.');
                return;
            }

            this.bindEvents();
        },

        bindEvents: function() {
            $('#image-upload').on('change', this.handleImageUpload.bind(this));
            $('#hairstyle-select').on('change', this.handleHairstyleChange.bind(this));
            $('#simulate-btn').on('click', this.submitRequest.bind(this));
            $('#try-another-btn').on('click', this.resetSimulator.bind(this));

            // User info fields - listen to both 'input' and 'change' for autofill support
            $('#aihs-full-name').on('input change', this.handleUserInfoChange.bind(this));
            $('#aihs-email').on('input change', this.handleUserInfoChange.bind(this));
            $('#aihs-phone').on('input change', this.handleUserInfoChange.bind(this));
            $('#aihs-city').on('input change', this.handleUserInfoChange.bind(this));
            $('#aihs-age').on('input change', this.handleUserInfoChange.bind(this));
            $('#aihs-consent').on('change', this.handleUserInfoChange.bind(this));

            // Modal open/close if present
            $('.ai-hair-simulator-open-modal').on('click', function() {
                $('#ai-hair-simulator-modal').addClass('is-open');
            });
            $(document).on('click', '.ai-hair-simulator-modal-close, .ai-hair-simulator-modal-backdrop', function() {
                $('#ai-hair-simulator-modal').removeClass('is-open');
            });

            // Check for autofill after a short delay (browsers autofill after page load)
            setTimeout(() => {
                this.checkAutofill();
            }, 100);

            // Also check on window load (for slower browsers)
            $(window).on('load', () => {
                setTimeout(() => {
                    this.checkAutofill();
                }, 200);
            });
        },

        checkAutofill: function() {
            // Check if fields have values (from autofill)
            const fullName = $('#aihs-full-name').val();
            const email = $('#aihs-email').val();
            
            if (fullName || email) {
                // Trigger validation if fields are filled
                this.handleUserInfoChange();
            }
        },

        handleUserInfoChange: function() {
            this.fullName = $('#aihs-full-name').val().trim();
            this.email = $('#aihs-email').val().trim();
            this.phone = $('#aihs-phone').length ? $('#aihs-phone').val().trim() : '';
            this.city = $('#aihs-city').length ? $('#aihs-city').val().trim() : '';
            this.age = $('#aihs-age').length ? $('#aihs-age').val().trim() : '';
            this.consentMarketing = $('#aihs-consent').is(':checked');
            
            // Add visual feedback for filled fields
            this.updateFieldVisuals();
            this.updateSimulateButton();
        },

        updateFieldVisuals: function() {
            // Full name validation
            if (this.fullName && this.fullName.length > 1) {
                $('#aihs-full-name').addClass('filled');
            } else {
                $('#aihs-full-name').removeClass('filled');
            }
            
            // Email validation
            if (this.email && this.validateEmail(this.email)) {
                $('#aihs-email').addClass('filled');
            } else {
                $('#aihs-email').removeClass('filled');
            }
            
            // Optional fields - just check if filled
            if (this.phone) $('#aihs-phone').addClass('filled');
            else $('#aihs-phone').removeClass('filled');
            
            if (this.city) $('#aihs-city').addClass('filled');
            else $('#aihs-city').removeClass('filled');
            
            if (this.age) $('#aihs-age').addClass('filled');
            else $('#aihs-age').removeClass('filled');
        },

        handleImageUpload: function(e) {
            const file = e.target.files[0];
            if (!file) {
                $('#image-upload').removeClass('filled');
                return;
            }

            // Validate file
            if (file.size > 3 * 1024 * 1024) {
                this.showError('Image size should not exceed 3MB.');
                $('#image-upload').removeClass('filled');
                return;
            }

            if (!['image/png', 'image/jpeg', 'image/jpg'].includes(file.type)) {
                this.showError('Invalid image format. Only PNG, JPG, JPEG are allowed.');
                $('#image-upload').removeClass('filled');
                return;
            }

            this.selectedImage = file;
            this.previewImageUrl = URL.createObjectURL(file);
            $('#preview-img').attr('src', this.previewImageUrl);
            $('#image-preview').show();
            $('#image-upload').addClass('filled');
            this.updateSimulateButton();
        },

        handleHairstyleChange: function(e) {
            this.selectedHairstyle = e.target.value;
        },

        updateSimulateButton: function() {
            const hasImage = this.selectedImage !== null;
            const hasName = this.fullName && this.fullName.length > 1;
            const hasEmail = this.email && this.validateEmail(this.email);
            const hasConsent = this.consentMarketing === true;
            $('#simulate-btn').prop('disabled', !(hasImage && hasName && hasEmail && hasConsent));
        },

        validateEmail: function(email) {
            // Simple email validation
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        },

        showError: function(message) {
            $('#error-message').text(message).show();
            setTimeout(() => {
                $('#error-message').fadeOut();
            }, 5000);
        },

        showLoading: function(title, message) {
            $('#loading-title').text(title);
            $('#loading-message').text(message);
            $('#ai-hair-simulator-loading').show();
        },

        hideLoading: function() {
            $('#ai-hair-simulator-loading').hide();
        },

        submitRequest: async function() {
            if (!this.selectedImage) {
                this.showError('Please select an image first.');
                return;
            }

            if (!this.fullName || !this.email || !this.validateEmail(this.email)) {
                this.showError('Please enter your full name and a valid email address.');
                return;
            }

            if (!this.consentMarketing) {
                this.showError('Please confirm that we can use your information to contact you.');
                return;
            }

            this.showLoading('Preparing...', 'Setting up your simulation');
            $('#ai-hair-simulator-upload').hide();
            $('#error-message').hide();

            try {
                // Step 1: Create simulation session
                const simulationResponse = await fetch(this.apiUrl + '/api/clinic-simulations', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        activationKey: this.activationKey,
                        website: this.website,
                        hairstyleSelected: this.selectedHairstyle,
                        originalImageUrl: null,
                        fullName: this.fullName,
                        email: this.email,
                        phone: this.phone,
                        city: this.city,
                        age: this.age,
                        consentMarketing: this.consentMarketing,
                    }),
                });

                if (!simulationResponse.ok) {
                    const errorData = await simulationResponse.json().catch(() => ({ statusMessage: 'Unknown error' }));
                    throw new Error(errorData.statusMessage || 'Failed to create simulation session');
                }

                const simulationResult = await simulationResponse.json();
                console.log('Simulation session created:', simulationResult);

                if (!simulationResult || !simulationResult.simulationId) {
                    throw new Error('Failed to receive simulation ID from the server.');
                }

                this.activeSimulationId = simulationResult.simulationId;

                // Step 2: Process hairstyle simulation
                this.showLoading('Processing...', 'AI is generating your new hairstyle');
                
                const formData = new FormData();
                formData.append('image', this.selectedImage);
                formData.append('hair_style', this.selectedHairstyle);
                if (this.activeSimulationId) {
                    formData.append('simulation_id', this.activeSimulationId);
                }

                const response = await fetch(this.apiUrl + '/api/hairstylesimulator', {
                    method: 'POST',
                    body: formData,
                });

                if (!response.ok) {
                    const errorData = await response.json().catch(() => ({ statusMessage: 'Unknown error' }));
                    throw new Error(errorData.statusMessage || errorData.message || 'Failed to process image');
                }

                const result = await response.json();
                console.log('Processing result:', result);

                // Handle immediate results
                if (result.immediate && result.imageUrl) {
                    this.processedImageUrl = result.imageUrl;
                    this.activeSimulationId = null;
                    this.showResults();
                    this.logSimulation();
                    return;
                }

                if (result.data && result.data.images && result.data.images.length > 0 && result.task_status === 2) {
                    this.processedImageUrl = result.data.images[0];
                    this.activeSimulationId = null;
                    this.showResults();
                    this.logSimulation();
                    return;
                }

                // Poll for results if task ID is provided
                if (result.taskId) {
                    this.pollForResult(result.taskId);
                } else {
                    throw new Error('No task ID or image URL received');
                }

            } catch (error) {
                console.error('Error:', error);
                this.hideLoading();
                $('#ai-hair-simulator-upload').show();
                this.showError(error.message || 'An error occurred. Please try again.');
            }
        },

        logSimulation: function() {
            if (typeof aiHairSimulator === 'undefined' || !aiHairSimulator.ajaxUrl || !aiHairSimulator.nonce) {
                return;
            }

            const payload = {
                action: 'ai_hair_simulator_log',
                nonce: aiHairSimulator.nonce,
                fullName: this.fullName,
                email: this.email,
                phone: this.phone,
                city: this.city,
                age: this.age,
                hairstyle: this.selectedHairstyle,
                processedImageUrl: this.processedImageUrl,
            };

            $.post(aiHairSimulator.ajaxUrl, payload)
                .done(function(response) {
                    if (!response || !response.success) {
                        console.warn('AI Hair Simulator: Failed to log simulation', response);
                    }
                })
                .fail(function(err) {
                    console.error('AI Hair Simulator: Error logging simulation', err);
                });
        },

        pollForResult: function(taskId) {
            let pollCount = 0;
            const maxPolls = 24;
            const pollInterval = 5000; // 5 seconds

            const intervalId = setInterval(async () => {
                pollCount++;

                try {
                    const response = await fetch(this.apiUrl + '/api/hairstylesimulator/status?taskId=' + taskId);
                    
                    if (!response.ok) {
                        throw new Error('Failed to check status');
                    }

                    const result = await response.json();

                    if (result.error) {
                        clearInterval(intervalId);
                        this.hideLoading();
                        $('#ai-hair-simulator-upload').show();
                        this.showError(result.error);
                        return;
                    }

                    if (result.task_status === 2) {
                        clearInterval(intervalId);
                        if (result.data && result.data.images && result.data.images.length > 0) {
                            this.processedImageUrl = result.data.images[0];
                            this.activeSimulationId = null;
                            this.showResults();
                            this.logSimulation();
                        } else {
                            this.hideLoading();
                            $('#ai-hair-simulator-upload').show();
                            this.showError('Processing complete, but no image was returned.');
                        }
                    } else if (pollCount >= maxPolls) {
                        clearInterval(intervalId);
                        this.hideLoading();
                        $('#ai-hair-simulator-upload').show();
                        this.showError('Processing is taking longer than expected. Please try again later.');
                    }

                } catch (error) {
                    console.error('Polling error:', error);
                    if (pollCount >= maxPolls) {
                        clearInterval(intervalId);
                        this.hideLoading();
                        $('#ai-hair-simulator-upload').show();
                        this.showError('Failed to fetch results. Please try again.');
                    }
                }
            }, pollInterval);

            // Timeout after 2 minutes
            setTimeout(() => {
                clearInterval(intervalId);
                if (this.processedImageUrl === null) {
                    this.hideLoading();
                    $('#ai-hair-simulator-upload').show();
                    this.showError('Processing timeout. Please try again.');
                }
            }, 120000);
        },

        showResults: function() {
            this.hideLoading();
            $('#ai-hair-simulator-results').show();
            
            // Set original image
            $('#original-image').attr('src', this.previewImageUrl);
            
            // Setup download button for processed image only
            $('#download-processed-btn').attr('href', this.processedImageUrl).show();
            
            // Handle processed image loading
            const processedImg = $('#processed-image');
            const loadingContainer = $('#processed-image-container .image-loading');
            
            // Check if image is already loaded (cached)
            if (processedImg[0] && processedImg[0].complete && processedImg[0].naturalWidth > 0) {
                loadingContainer.hide();
                processedImg.show();
            } else {
                // Set image source
                processedImg.attr('src', this.processedImageUrl);
                
                // Hide loading when image loads
                processedImg.on('load', function() {
                    loadingContainer.hide();
                    processedImg.show();
                }).on('error', function() {
                    loadingContainer.hide();
                    HairSimulator.showError('Failed to load processed image. Please try again.');
                });
            }
        },

        resetSimulator: function() {
            this.selectedImage = null;
            this.previewImageUrl = null;
            this.processedImageUrl = null;
            this.activeSimulationId = null;
            this.selectedHairstyle = 'autodetect';
            
            $('#image-upload').val('');
            $('#hairstyle-select').val('autodetect');
            $('#image-preview').hide();
            $('#ai-hair-simulator-results').hide();
            $('#ai-hair-simulator-upload').show();
            $('#simulate-btn').prop('disabled', true);
            $('#download-processed-btn').hide();
        }
    };

    $(document).ready(function() {
        HairSimulator.init();
    });

})(jQuery);

